function CalculateTotalHours(cards) {
  let totalHours = 0;
  let cardsWithHours = 0;
  let hasFormatIssues = false;
  let problematicCards = [];
  let problematicCardElements = [];
  let formatIssues = []; // Store detailed format issues

  cards.forEach((card) => {
    // Look for estimated hours in the new Trello structure
    const timeEls = card.querySelectorAll(".QipMQPdkdydS8I");
    let foundHour = false;
    let cardHasIssue = false;
    let cardFormatIssue = null;
    let hasTimeField = false; // Track if card has any time field

    // Get card name for reporting
    const cardNameEl = card.querySelector('a[data-testid="card-name"]');
    const cardName = cardNameEl ? cardNameEl.textContent.trim() : 'Onbekende kaart';

    console.log("Processing card:", cardName, "with", timeEls.length, "time elements");

    timeEls.forEach((timeEl) => {
      const textEl = timeEl.querySelector(".nGT0DJOrI676qn");
      if (!textEl) return;
      
      const text = textEl.textContent.trim();
      console.log("Found text:", text, "for card:", cardName); // Debug log

      // Check for both "estimated hours:" and "inschatting:" formats
      const isEstimatedHours = text.toLowerCase().startsWith("estimated hours:");
      const isInschatting = text.toLowerCase().startsWith("inschatting:");
      
      console.log("isEstimatedHours:", isEstimatedHours, "isInschatting:", isInschatting, "for card:", cardName);
      
      if (!isEstimatedHours && !isInschatting) return;

      // Mark that we found a time field
      hasTimeField = true;

      let timeText = text
        .replace(/estimated hours:/i, "")
        .replace(/inschatting:/i, "")
        .trim()
        .toLowerCase()
        .replace(",", ".");

      console.log("Processed timeText:", timeText, "for card:", cardName);
      console.log("timeText length:", timeText.length, "characters:", Array.from(timeText).map(c => c.charCodeAt(0)));

      // HH:MM format (e.g. 1:00, 2:30, 0:00) - treat as decimal hours
      const timeMatch = timeText.match(/^(\d{1,2}):(\d{2})$/);
      if (timeMatch) {
        console.log("Found HH:MM format:", timeText, "for card:", cardName);
        const hours = parseInt(timeMatch[1], 10);
        const minutes = parseInt(timeMatch[2], 10);
        
        console.log("Parsed HH:MM - hours:", hours, "minutes:", minutes, "for card:", cardName);
        
        // Special debug for 0:00 format
        if (hours === 0 && minutes === 0) {
          console.log("Found 0:00 format for card:", cardName, "- this should be valid");
        }
        
        // Validate minutes (should be 0-59)
        if (minutes >= 60) {
          console.log("Invalid minutes in HH:MM format:", timeText, "for card:", cardName);
          hasFormatIssues = true;
          cardHasIssue = true;
          
          cardFormatIssue = {
            cardName: cardName,
            originalText: text,
            inputValue: timeText,
            suggestedFormat: `${hours}:${minutes.toString().padStart(2, '0')}`,
            issue: `Ongeldige minuten in "${timeText}". Minuten moeten tussen 0-59 zijn.`
          };
        } else {
          // Convert to decimal hours (e.g., 2:30 = 2.5 hours, 0:45 = 0.75 hours)
          const calculatedHours = hours + (minutes / 60);
          totalHours += calculatedHours;
          foundHour = true;
          console.log("Valid HH:MM format processed:", timeText, "->", calculatedHours, "hours for card:", cardName);
        }
        return;
      }

      // Check for HH:MM format with h suffix (e.g., 2:50h) - check this BEFORE incorrectFormat
      const timeWithSuffix = timeText.match(/^(\d{1,2}):(\d{2})(u|h)$/i);
      console.log('timeWithSuffix check for', timeText, ':', timeWithSuffix);
      console.log('timeWithSuffix regex test:', /^(\d{1,2}):(\d{2})(u|h)$/i.test(timeText));
      if (timeWithSuffix) {
        console.log('Found HH:MM with suffix format:', timeText, 'in card:', cardName);
        console.log('timeWithSuffix match:', timeWithSuffix);
        hasFormatIssues = true;
        cardHasIssue = true;
        foundHour = true;
        
        const hours = parseInt(timeWithSuffix[1], 10);
        const minutes = parseInt(timeWithSuffix[2], 10);
        const unit = timeWithSuffix[3];
        
        console.log('Parsed timeWithSuffix:', { hours, minutes, unit });
        
        // Validate minutes
        if (minutes >= 60) {
          cardFormatIssue = {
            cardName: cardName,
            originalText: text,
            inputValue: timeText,
            suggestedFormat: `${hours}:${minutes.toString().padStart(2, '0')}`,
            issue: `Ongeldige minuten in "${timeText}". Minuten moeten tussen 0-59 zijn.`
          };
        } else {
          // Convert to decimal hours (e.g., 2:30 = 2.5 hours, 0:45 = 0.75 hours)
          totalHours += hours + (minutes / 60);
          
          cardFormatIssue = {
            cardName: cardName,
            originalText: text,
            inputValue: timeText,
            suggestedFormat: `${hours}:${minutes.toString().padStart(2, '0')}`,
            issue: `Je gebruikt "${timeText}" met "${unit}" aan het einde. Gebruik het HH:MM formaat in plaats daarvan.`
          };
        }
        console.log('Created cardFormatIssue for timeWithSuffix:', cardFormatIssue);
        return; // Add return here to prevent fallthrough
      }

      // Check for incorrect formats (with u or h) - but NOT HH:MM format
      const incorrectFormat = timeText.match(/^(\d*\.\d+|\d+)(u|h)$/);
      if (incorrectFormat) {
        console.log('Found incorrect format:', timeText, 'in card:', cardName);
        hasFormatIssues = true;
        cardHasIssue = true;
        foundHour = true;
        
        // Store detailed format issue
        const value = incorrectFormat[1];
        const unit = incorrectFormat[2];
        
        // Add the time to total (treat as hours)
        const timeValue = parseFloat(value);
        totalHours += timeValue;
        console.log("Incorrect format processed:", timeText, "->", timeValue, "hours for card:", cardName);
        
        // Convert to HH:MM format for better suggestion
        let suggestedFormat = value;
        if (value.includes('.')) {
          // Convert decimal to HH:MM (e.g., 1.5 -> 1:30)
          const hours = Math.floor(parseFloat(value));
          const minutes = Math.round((parseFloat(value) - hours) * 60);
          suggestedFormat = `${hours}:${minutes.toString().padStart(2, '0')}`;
        } else {
          // Convert whole number to HH:MM (e.g., 2 -> 2:00)
          suggestedFormat = `${value}:00`;
        }
        
        cardFormatIssue = {
          cardName: cardName,
          originalText: text,
          inputValue: timeText,
          suggestedFormat: suggestedFormat,
          issue: `Je gebruikt "${timeText}" met "${unit}" aan het einde. Gebruik het HH:MM formaat in plaats daarvan.`
        };
        return; // Add return here to prevent fallthrough
      }

      // Numeric format without unit (assume hours) - this is correct format
      const numericMatch = timeText.match(/^(\d*\.\d+|\d+)$/);
      if (numericMatch) {
        const value = parseFloat(numericMatch[1]);
        totalHours += value;
        foundHour = true;
        console.log("Valid numeric format processed:", timeText, "->", value, "hours for card:", cardName);
        return;
      }

      // If we reach here, the time format is not recognized
      console.log("Unrecognized time format:", timeText, "for card:", cardName);
      hasFormatIssues = true;
      cardHasIssue = true;
      
      cardFormatIssue = {
        cardName: cardName,
        originalText: text,
        inputValue: timeText,
        suggestedFormat: '0:00',
        issue: `Onbekend tijdformaat: "${timeText}". Gebruik HH:MM (bijv. 1:30) of alleen cijfers (bijv. 1.5).`
      };
    });

    console.log("Card processing result for", cardName, "- foundHour:", foundHour, "hasTimeField:", hasTimeField, "cardHasIssue:", cardHasIssue);
    
    if (foundHour) {
      cardsWithHours++;
      console.log("Card", cardName, "has valid time format - added to cardsWithHours");
    } else if (!hasTimeField) {
      // Card has no "Inschatting:" or "Estimated hours:" field at all - ignore this card
      console.log("Ignoring card without time field:", cardName);
      
      // Add "ignored" tag to cards that have other badges but no time field
      const badgeContainer = card.querySelector('.b5UcmsGnhjD0Go');
      if (badgeContainer) {
        // Check if there's already an ignored tag
        const existingIgnoredTag = badgeContainer.querySelector('.ignored-time-tag');
        if (!existingIgnoredTag) {
          const ignoredTag = document.createElement('span');
          ignoredTag.className = 'ignored-time-tag QipMQPdkdydS8I';
          ignoredTag.style.cssText = `
                color: var(--ds-text, #172b4d);
            font-size: 11px;
            padding: 2px 6px;
            border-radius: 3px;
          `;
          ignoredTag.textContent = 'Geen inschatting';
          ignoredTag.title = 'Deze kaart heeft geen tijdnotatie en wordt genegeerd';
          badgeContainer.appendChild(ignoredTag);
        }
      }
    } else {
      console.log("Card", cardName, "has time field but no valid time format found");
    }

    // Add format issues if they exist (regardless of foundHour status)
    if (cardFormatIssue) {
      // Ignore cards with "Sprintmeeting" in the name
      if (!cardName.toLowerCase().includes('sprintmeeting')) {
        console.log("Adding format issue for card:", cardName, cardFormatIssue);
        formatIssues.push(cardFormatIssue);
      } else {
        console.log("Ignoring Sprintmeeting card:", cardName);
      }
    }
  });

  console.log("Total hours calculated:", totalHours, "Cards with hours:", cardsWithHours, "Has format issues:", hasFormatIssues, "Problematic cards:", problematicCards, "Format issues:", formatIssues); // Debug log
  
  // Ensure hasFormatIssues is set correctly based on actual format issues
  const hasAnyFormatIssues = formatIssues.length > 0;
  
  // Convert total hours back to HH:MM format for display
  const displayHours = Math.floor(totalHours);
  const displayMinutes = Math.round((totalHours - displayHours) * 60);
  const displayTime = `${displayHours}:${displayMinutes.toString().padStart(2, '0')}`;
  
  return {
    hours: displayTime,
    hasMissing: cardsWithHours < cards.length,
    hasFormatIssues: hasAnyFormatIssues,
    problematicCards: problematicCards,
    problematicCardElements: problematicCardElements,
    formatIssues: formatIssues,
  };
}

function createOrUpdateListBadge(headerEl, cardCount, cards) {
  // Remove collapse button to save space
  const collapseBtn = headerEl.querySelector(
    'button[data-testid="list-collapse-button"]',
  );
  if (collapseBtn) collapseBtn.remove();

  // Get parent list container
  const listContainer = headerEl.closest('[data-testid="list"]');
  if (!listContainer) return;

  // Check for or create badge wrapper
  let wrapper = listContainer.querySelector(".list-time-flex-wrapper");
  if (!wrapper) {
    wrapper = document.createElement("div");
    wrapper.className = "list-time-flex-wrapper";
    wrapper.style.cssText = `
      display: flex;
      align-items: center;
      gap: 8px;
      padding: 8px 8px 0;
      justify-content: space-between;
      flex-wrap: wrap;
    `;
    const afterHeader = headerEl.nextSibling;
    listContainer.insertBefore(wrapper, afterHeader);
  }

  // Remove any previous custom label
  const oldCustomLabel = wrapper.querySelector(".custom-time-label");
  if (oldCustomLabel) oldCustomLabel.remove();

  // set .e5LZFj7edvnuic display none
  const e5LZFj7edvnuic = listContainer.querySelector(".e5LZFj7edvnuic");
  if (e5LZFj7edvnuic) e5LZFj7edvnuic.style.display = "none";

  // Add new label
  const label = document.createElement("p");
  label.className = "custom-time-label";
  label.style.cssText = `
    margin: 2px 0;
    padding-left: 12px;
    color: var(--accent-text, var(--ds-text-subtlest, #626f86));
    line-height: 16px;
  `;
  label.textContent = `${cardCount} Kaart${cardCount === 1 ? "" : "en"}`;
  wrapper.appendChild(label);

  const { hours, hasMissing, hasFormatIssues, problematicCards, problematicCardElements, formatIssues } = CalculateTotalHours(cards);

  // Create a container div for the badges
  const badgeContainer = document.createElement("div");
  badgeContainer.style.cssText = `
    display: flex;
    align-items: center;
    gap: 4px;
    margin-left: auto;
  `;

  const badge = document.createElement("span");
  badge.className = "total-hours-badge";
  badge.style.cssText = `
      background: ${hasFormatIssues ? "#d62828" : "#44546f"};
      color: white;
      font-size: 12px;
      font-weight: bold;
      padding: 0.3rem 0.6rem;
      border-radius: 6px;
      border: none;
      cursor: default;
      line-height: 1;
      flex-shrink: 0;
    `;
  badge.textContent = hours;
  badgeContainer.appendChild(badge);

  // Add warning badge if there are format issues
  if (hasFormatIssues) {
    const warningBadge = document.createElement("span");
    warningBadge.className = "format-warning-badge";
    
    // Get Trello's card color (use the first problematic card's color)
    let cardColor = '#d62828'; // default red
    if (problematicCardElements.length > 0) {
      const firstCard = problematicCardElements[0];
      const labelEl = firstCard.querySelector('.CNnougcEO0WaO3');
      if (labelEl) {
        const colorAttr = labelEl.getAttribute('data-color');
        if (colorAttr) {
          // Map Trello colors to hex values
          const colorMap = {
            'black_light': '#42526e',
            'blue': '#0079bf',
            'green': '#61bd4f',
            'lime': '#51e898',
            'orange': '#ff9f1a',
            'pink': '#ff78cb',
            'purple': '#c377e0',
            'red': '#eb5a46',
            'sky': '#00c2e0',
            'yellow': '#f2d600'
          };
          cardColor = colorMap[colorAttr] || '#d62828';
        }
      }
    }
    
    warningBadge.style.cssText = `
      background: ${cardColor};
      color: white;
      font-size: 12px;
      font-weight: bold;
      padding: 0.3rem 0.6rem;
      border-radius: 6px;
      border: none;
      cursor: default;
      line-height: 1;
      flex-shrink: 0;
      position: relative;
      animation: warningPulse 5s infinite;
    `;
    warningBadge.textContent = "!";
    warningBadge.title = "Er zijn kaarten met een onjuiste tijdnotatie in deze lijst.";
    badgeContainer.appendChild(warningBadge);
  }

  wrapper.appendChild(badgeContainer);
}

function calculateCardCountPerList() {
  // Remove previous badges
  document
    .querySelectorAll(".list-time-flex-wrapper")
    .forEach((el) => el.remove());

  // Hide .JGEXNtVeatgEgC elements from every list using CSS instead of removing them
  document
    .querySelectorAll(".JGEXNtVeatgEgC")
    .forEach((el) => {
      try {
        if (el) {
          el.style.display = "none";
        }
      } catch (error) {
        console.log("Error hiding .JGEXNtVeatgEgC element:", error);
      }
    });

  const lists = document.querySelectorAll('li[data-testid="list-wrapper"]');

  lists.forEach((list) => {
    const header = list.querySelector('div[data-testid="list-header"]');
    const cards = list.querySelectorAll(
      'li[data-testid="list-card"]:not([hidden])',
    );
    const cardCount = cards.length;

    createOrUpdateListBadge(header, cardCount, cards);
  });
  
  // Update navigation error icon
  updateNavigationErrorIcon();
}

function updateNavigationErrorIcon() {
  // Find the navigation panel
  const navPanel = document.querySelector('[data-testid="panel-nav-board-button"]');
  if (!navPanel) return;

  // Check if there are any format issues across all lists
  const allCards = document.querySelectorAll('li[data-testid="list-card"]:not([hidden])');
  const { hasFormatIssues, problematicCards, problematicCardElements, formatIssues } = CalculateTotalHours(allCards);

  // Find or create the error/info icon button
  let errorIcon = document.querySelector('.trello-hours-error-icon');
  if (!errorIcon) {
    errorIcon = document.createElement('button');
    errorIcon.className = [
      'trello-hours-error-icon',
      'APu9_M1MYhkJun',
      'bxgKMAm3lq5BpA',
      'iUcMblFAuq9LKn',
      'PnEv2xIWy3eSui'
    ].join(' ');
    errorIcon.type = 'button';
    errorIcon.style.cssText = `
      margin-left: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      min-width: 32px;
      height: 32px;
      font-size: 16px;
      transition: all 0.2s;
      position: relative;
      background: none;
      border: none;
      cursor: pointer;
      border-radius: 4px;
      box-shadow: none;
    `;
    // Insert after the board button
    navPanel.parentElement.parentElement.appendChild(errorIcon);
  }

  // Store current state to avoid unnecessary updates
  const currentIcon = errorIcon.innerHTML;
  const currentColor = errorIcon.style.color;
  const newIcon = hasFormatIssues ? '⚠️' : '✅';
  const newColor = hasFormatIssues ? '#eb5a46' : '#0079bf';
  
  // Only update if something changed
  if (currentIcon !== newIcon || currentColor !== newColor) {
    errorIcon.innerHTML = newIcon;
    errorIcon.style.color = newColor;
    
    if (hasFormatIssues) {
      errorIcon.title = `${problematicCards.length} kaart${problematicCards.length === 1 ? '' : 'en'} met tijdnotatie problemen`;
      
      // Remove existing event listeners
      errorIcon.onclick = null;
      errorIcon.onmouseenter = null;
      errorIcon.onmouseleave = null;
      
      // Add new event listeners
      errorIcon.onclick = (e) => {
        e.preventDefault();
        e.stopPropagation();
        // Remove existing popup if any
        const existingPopup = document.querySelector('.format-issues-popup');
        if (existingPopup) {
          existingPopup.remove();
          return;
        }
        createFormatIssuesPopup(problematicCardElements, formatIssues);
      };
      
      errorIcon.onmouseenter = () => {
        errorIcon.style.background = 'rgba(235, 90, 70, 0.1)';
        errorIcon.style.transform = 'scale(1.1)';
      };
      errorIcon.onmouseleave = () => {
        errorIcon.style.background = 'none';
        errorIcon.style.transform = 'scale(1)';
      };
    } else {
      errorIcon.title = 'Geen tijdnotatie problemen';
      
      // Remove existing event listeners
      errorIcon.onclick = null;
      errorIcon.onmouseenter = null;
      errorIcon.onmouseleave = null;
      
      // Add new event listeners
      errorIcon.onclick = (e) => {
        e.preventDefault();
        e.stopPropagation();
        // Remove existing info popup if any
        const existingPopup = document.querySelector('.trello-hours-no-issues-popup');
        if (existingPopup) {
          existingPopup.remove();
          return;
        }
        // Show info popup
        const infoPopup = document.createElement('div');
        infoPopup.className = 'trello-hours-no-issues-popup';
        infoPopup.style.cssText = `
          position: fixed;
          bottom: 70px;
          left: 28px;
          background: #fff;
          border: 1px solid #dfe1e6;
          border-radius: 8px;
          box-shadow: 0 4px 16px rgba(9,30,66,.15);
          padding: 16px 24px;
          z-index: 10001;
          font-size: 15px;
          color: #172b4d;
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          animation: slideInUp 0.3s ease-out;
        `;
        infoPopup.textContent = 'Er zijn geen kaarten met tijdnotatie problemen.';
        document.body.appendChild(infoPopup);
        setTimeout(() => {
          if (infoPopup.parentElement) infoPopup.remove();
        }, 3000);
      };
      
      errorIcon.onmouseenter = () => {
        errorIcon.style.background = 'rgba(0, 123, 191, 0.08)';
        errorIcon.style.transform = 'scale(1.1)';
      };
      errorIcon.onmouseleave = () => {
        errorIcon.style.background = 'none';
        errorIcon.style.transform = 'scale(1)';
      };
    }
  }
}

function createFormatIssuesPopup(problematicCardElements, formatIssues) {
  // Create accordion popup
  const popup = document.createElement('div');
  popup.className = 'format-issues-popup';
  popup.style.cssText = `
    position: fixed;
    bottom: 20px;
    left: 20px;
    background: #fff;
    border: 1px solid #dfe1e6;
    border-radius: 8px;
    box-shadow: 0 4px 16px rgba(9,30,66,.25);
    padding: 0;
    max-width: 420px;
    min-width: 320px;
    max-height: 80vh;
    overflow-y: auto;
    z-index: 10000;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    animation: slideInUp 0.5s ease-out;
  `;
  
  // Overlay for closing
  const overlay = document.createElement('div');
  overlay.style.cssText = `
    position: fixed;
    top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(9,30,66,.15);
    z-index: 9999;
  `;
  overlay.addEventListener('click', () => {
    popup.remove();
    overlay.remove();
  });
  document.body.appendChild(overlay);
  
  // Popup header
  const header = document.createElement('div');
  header.style.cssText = `
    display: flex;
    align-items: center;
    justify-content: space-between;
    background: #f4f5f7;
    border-bottom: 1px solid #dfe1e6;
    border-radius: 8px 8px 0 0;
    padding: 16px 20px;
  `;
  
  const headerContent = document.createElement('div');
  headerContent.style.cssText = 'display: flex; align-items: center;';
  
  const headerIcon = document.createElement('div');
  headerIcon.style.cssText = `
    width: 24px;
    height: 24px;
    border-radius: 50%;
    margin-right: 12px;
    background-image: url('https://avatars.githubusercontent.com/u/166696801?s=200&v=4');
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
  `;
  
  const title = document.createElement('span');
  title.textContent = '⚠️ Tijdnotatie Problemen';
  title.style.cssText = 'font-size: 16px; color: #172b4d; font-weight: 600;';
  
  const closeBtn = document.createElement('button');
  closeBtn.innerHTML = '&times;';
  closeBtn.setAttribute('aria-label', 'Sluiten');
  closeBtn.style.cssText = `
    background: none;
    border: none;
    font-size: 24px;
    color: #6b778c;
    cursor: pointer;
    margin-left: 8px;
    line-height: 1;
  `;
  closeBtn.addEventListener('click', () => {
    popup.remove();
    overlay.remove();
  });
  
  headerContent.appendChild(headerIcon);
  headerContent.appendChild(title);
  header.appendChild(headerContent);
  header.appendChild(closeBtn);
  popup.appendChild(header);
  
  // Info text
  const info = document.createElement('div');
  info.textContent = 'De volgende kaarten hebben een onjuiste tijdnotatie:';
  info.style.cssText = 'padding: 16px 20px 0 20px; color: #5e6c84; font-size: 14px;';
  popup.appendChild(info);
  
  // Group cards by list - include ALL cards with format issues
  const listsWithIssues = new Map();
  
  console.log("Total problematic cards:", problematicCardElements.length);
  console.log("Total format issues:", formatIssues.length);
  
  // Create a set of all card names with format issues
  const cardsWithFormatIssues = new Set(formatIssues.map(issue => issue.cardName));
  console.log("Cards with format issues:", Array.from(cardsWithFormatIssues));
  
  // Find all cards on the board that have format issues
  const allCards = document.querySelectorAll('li[data-testid="list-card"]:not([hidden])');
  const cardsToShow = [];
  
  allCards.forEach((card) => {
    const cardNameEl = card.querySelector('a[data-testid="card-name"]');
    const cardName = cardNameEl ? cardNameEl.textContent.trim() : 'Onbekende kaart';
    
    if (cardsWithFormatIssues.has(cardName)) {
      cardsToShow.push(card);
    }
  });
  
  console.log("Found", cardsToShow.length, "cards to show in popup");
  
  cardsToShow.forEach((card) => {
    const listContainer = card.closest('[data-testid="list"]');
    if (listContainer) {
      const listNameEl = listContainer.querySelector('h2[data-testid="list-name"]');
      const listName = listNameEl ? listNameEl.textContent.trim() : 'Onbekende lijst';
      
      if (!listsWithIssues.has(listName)) {
        listsWithIssues.set(listName, []);
      }
      listsWithIssues.get(listName).push(card);
    }
  });
  
  console.log("Lists with issues:", Array.from(listsWithIssues.keys()));
  console.log("Cards per list:", Array.from(listsWithIssues.entries()).map(([list, cards]) => `${list}: ${cards.length}`));
  
  // Create accordion content
  const accordionContainer = document.createElement('div');
  accordionContainer.style.cssText = `
    padding: 8px 20px 0 20px;
    max-height: 300px;
    overflow-y: auto;
  `;
  
  console.log("Creating accordion for", listsWithIssues.size, "lists");
  
  listsWithIssues.forEach((cards, listName) => {
    console.log("Creating accordion item for list:", listName, "with", cards.length, "cards");
    // List accordion item
    const listItem = document.createElement('div');
    listItem.style.cssText = `
      margin-bottom: 8px;
      border: 1px solid #dfe1e6;
      border-radius: 6px;
      overflow: hidden;
    `;
    
    // List header (clickable)
    const listHeader = document.createElement('div');
    listHeader.style.cssText = `
      background: #f8f9fa;
      padding: 12px 16px;
      cursor: pointer;
      display: flex;
      align-items: center;
      justify-content: space-between;
      border-bottom: 1px solid #dfe1e6;
      transition: background-color 0.2s;
    `;
    
    const listTitle = document.createElement('div');
    listTitle.style.cssText = `
      font-weight: 600;
      color: #172b4d;
      font-size: 14px;
    `;
    listTitle.textContent = `${listName} (${cards.length} kaart${cards.length === 1 ? '' : 'en'})`;
    
    const expandIcon = document.createElement('span');
    expandIcon.innerHTML = '▼';
    expandIcon.style.cssText = `
      color: #6b778c;
      font-size: 12px;
      transition: transform 0.2s;
    `;
    
    listHeader.appendChild(listTitle);
    listHeader.appendChild(expandIcon);
    
    // Cards container (initially hidden)
    const cardsContainer = document.createElement('div');
    cardsContainer.style.cssText = `
      background: #fff;
      max-height: 0;
      overflow: hidden;
      transition: max-height 0.3s ease-out;
    `;
    
    // Add cards to container
    cards.forEach((card) => {
      const cardNameEl = card.querySelector('a[data-testid="card-name"]');
      const cardName = cardNameEl ? cardNameEl.textContent.trim() : 'Onbekende kaart';
      
      // Get format issue details for this card
      const formatIssue = formatIssues.find(issue => issue.cardName === cardName);
      
      // Get card color
      let cardColor = '#d62828';
      const labelEl = card.querySelector('.CNnougcEO0WaO3');
      if (labelEl) {
        const colorAttr = labelEl.getAttribute('data-color');
        if (colorAttr) {
          const colorMap = {
            'black_light': '#42526e',
            'blue': '#0079bf',
            'green': '#61bd4f',
            'lime': '#51e898',
            'orange': '#ff9f1a',
            'pink': '#ff78cb',
            'purple': '#c377e0',
            'red': '#eb5a46',
            'sky': '#00c2e0',
            'yellow': '#f2d600'
          };
          cardColor = colorMap[colorAttr] || '#d62828';
        }
      }
      
      // Create card wrapper
      const cardWrapper = document.createElement('div');
      cardWrapper.style.cssText = `
        background: #fff;
        border-bottom: 1px solid #f4f5f7;
        padding: 12px 16px;
        cursor: pointer;
        transition: all 0.2s ease;
        position: relative;
        overflow: hidden;
      `;
      
      // Add hover effect
      cardWrapper.addEventListener('mouseenter', () => {
        cardWrapper.style.background = '#f8f9fa';
      });
      
      cardWrapper.addEventListener('mouseleave', () => {
        cardWrapper.style.background = '#fff';
      });
      
      // Add click handler to open card
      cardWrapper.addEventListener('click', (ev) => {
        ev.preventDefault();
        ev.stopPropagation();
        
        // Close the popup first
        const popup = document.querySelector('.format-issues-popup');
        const overlay = document.querySelector('.format-issues-popup + div');
        if (popup) popup.remove();
        if (overlay) overlay.remove();
        
        // Try multiple ways to open the card
        let cardOpened = false;
        
        if (cardNameEl) {
          cardNameEl.click();
          cardOpened = true;
        } else {
          const allCards = document.querySelectorAll('a[data-testid="card-name"]');
          for (let cardLink of allCards) {
            if (cardLink.textContent.trim() === cardName) {
              cardLink.click();
              cardOpened = true;
              break;
            }
          }
        }
        
        if (!cardOpened) {
          const cardId = card.getAttribute('data-card-id');
          if (cardId) {
            const cardById = document.querySelector(`[data-card-id="${cardId}"]`);
            if (cardById) {
              const cardLink = cardById.querySelector('a[data-testid="card-name"]');
              if (cardLink) {
                cardLink.click();
                cardOpened = true;
              }
            }
          }
        }
        
        if (!cardOpened) {
          const quickEditBtn = card.querySelector('button[data-testid="quick-card-editor-button"]');
          if (quickEditBtn) {
            quickEditBtn.click();
            cardOpened = true;
          }
        }
      });
      

      
      // Color indicator
      if (cardColor !== '#d62828') {
        const colorIndicator = document.createElement('div');
        colorIndicator.style.cssText = `
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
          height: 2px;
          background: ${cardColor};
        `;
        cardWrapper.appendChild(colorIndicator);
      }
      
      // Card content
      const cardContent = document.createElement('div');
      cardContent.style.cssText = 'position: relative;';
      
      // Card title
      const cardTitle = document.createElement('div');
      cardTitle.textContent = cardName;
      cardTitle.style.cssText = `
        font-size: 13px;
        line-height: 18px;
        color: #172b4d;
        font-weight: 500;
        word-wrap: break-word;
        margin-bottom: 4px;
      `;
      
      // Error details
      const errorDetails = document.createElement('div');
      errorDetails.style.cssText = `
        font-size: 11px;
        color: #eb5a46;
        line-height: 14px;
      `;
      
      if (formatIssue) {
        errorDetails.innerHTML = `
          <div style="margin-bottom: 2px;">
            <strong>Huidige:</strong> "${formatIssue.inputValue}"
          </div>
          <div style="color: #61bd4f; font-weight: 500;">
            <strong>Gebruik:</strong> "${formatIssue.suggestedFormat}"
          </div>
        `;
      } else {
        errorDetails.innerHTML = `
          <span style="margin-right: 4px;">⚠️</span>
          Onjuiste tijdnotatie
        `;
      }
      
      cardContent.appendChild(cardTitle);
      cardContent.appendChild(errorDetails);
      cardWrapper.appendChild(cardContent);
      cardsContainer.appendChild(cardWrapper);
    });
    
    // Toggle functionality
    let isExpanded = false;
    listHeader.addEventListener('click', () => {
      isExpanded = !isExpanded;
      
      if (isExpanded) {
        cardsContainer.style.maxHeight = '400px';
        expandIcon.style.transform = 'rotate(180deg)';
        listHeader.style.background = '#e3f2fd';
      } else {
        cardsContainer.style.maxHeight = '0';
        expandIcon.style.transform = 'rotate(0deg)';
        listHeader.style.background = '#f8f9fa';
      }
    });
    
    listItem.appendChild(listHeader);
    listItem.appendChild(cardsContainer);
    accordionContainer.appendChild(listItem);
  });
  
  console.log("Adding accordion container to popup with", accordionContainer.children.length, "list items");
  popup.appendChild(accordionContainer);
  
  // Solution box
  const solution = document.createElement('div');
  solution.style.cssText = `
    background: #f0f8ff;
    border: 1px solid #0079bf;
    border-radius: 6px;
    padding: 12px 20px;
    margin: 20px;
    margin-bottom: 16px;
    color: #0079bf;
    font-size: 14px;
  `;
  solution.innerHTML = '<div style="font-weight: bold; color: #0079bf; margin-bottom: 4px;">💡 Oplossing:</div>Gebruik het juiste formaat: <strong>HH:MM</strong> (bijv. 1:30) of alleen cijfers (bijv. 1.5 voor 1,5 uur)';
  popup.appendChild(solution);
  
  // Credits
  const credits = document.createElement('div');
  credits.style.cssText = `
    text-align: center;
    padding: 8px 20px 16px 20px;
    color: #6b778c;
    font-size: 11px;
    border-top: 1px solid #f4f5f7;
    margin-top: 8px;
  `;
  credits.textContent = 'Made with ❤️ by Jelte';
  popup.appendChild(credits);
  
  document.body.appendChild(popup);
}

let timeout = null;
let lastUpdateTime = 0;
const UPDATE_THROTTLE = 2000; // Only update every 2 seconds

function waitForBoardAndInit() {
  // Add CSS animations if they don't exist
  if (!document.querySelector('#trello-hours-extension-styles')) {
    const style = document.createElement('style');
    style.id = 'trello-hours-extension-styles';
    style.textContent = `
      @keyframes warningPulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.7; }
      }
      
      @keyframes slideInUp {
        from {
          transform: translateY(100%);
          opacity: 0;
        }
        to {
          transform: translateY(0);
          opacity: 1;
        }
      }
    `;
    document.head.appendChild(style);
  }

  const board = document.querySelector("#board");
  const hasLists =
    document.querySelectorAll('li[data-testid="list-wrapper"]').length > 0;

  if (board && hasLists) {
    calculateCardCountPerList();

    const observer = new MutationObserver(() => {
      const now = Date.now();
      if (now - lastUpdateTime > UPDATE_THROTTLE) {
        clearTimeout(timeout);
        timeout = setTimeout(() => {
          calculateCardCountPerList();
          lastUpdateTime = now;
        }, 100);
      }
    });

    observer.observe(board, {
      childList: true,
      subtree: true,
    });
  } else {
    setTimeout(waitForBoardAndInit, 1000);
  }
}

// Remove the second observer that was causing double updates
waitForBoardAndInit();

// Show welcome/update notification
function showWelcomeNotification() {
  // Check if user has seen this version
  const currentVersion = '2.1';
  const lastSeenVersion = localStorage.getItem('trello-hours-extension-version');
  
  if (lastSeenVersion === currentVersion) return;
  
  setTimeout(() => {
    const notification = document.createElement('div');
    notification.className = 'welcome-notification';
    notification.style.cssText = `
      position: fixed;
      bottom: 20px;
      left: 20px;
      background: #fff;
      border: 1px solid #dfe1e6;
      border-radius: 8px;
      box-shadow: 0 4px 16px rgba(9,30,66,.25);
      padding: 16px;
      max-width: 320px;
      z-index: 10000;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      animation: slideInUp 0.5s ease-out;
    `;
    
    notification.innerHTML = `
      <div style="display: flex; align-items: center; margin-bottom: 12px;">
        <div style="
          width: 32px;
          height: 32px;
          border-radius: 50%;
          margin-right: 12px;
          background-image: url('https://avatars.githubusercontent.com/u/166696801?s=200&v=4');
          background-size: cover;
          background-position: center;
          background-repeat: no-repeat;
        "></div>
        <div>
          <div style="font-weight: 600; color: #172b4d; font-size: 14px;">Trello Hours Extension</div>
          <div style="color: #6b778c; font-size: 12px;">v${currentVersion} - Nieuw!</div>
        </div>
      </div>
      
      <div style="color: #172b4d; font-size: 13px; line-height: 1.4; margin-bottom: 12px;">
        <div style="margin-bottom: 8px;"><strong>🎉 Nieuwe functies:</strong></div>
        <div style="margin-bottom: 4px;">• Ondersteunt nu <strong>"Inschatting:"</strong> en <strong>"Estimated Hours:"</strong></div>
        <div style="margin-bottom: 4px;">• Werkt op alle Trello boards</div>
        <div style="margin-bottom: 4px;">• Verbeterde tijdnotatie detectie</div>
        <div style="margin-bottom: 4px;">• Klikbare waarschuwingen voor foute formaten</div>

        <div>• <strong>Nieuwe:</strong> Accordion popup met lijst-overzicht</div>
      </div>
      
      <div style="
        background: #f4f5f7;
        border-radius: 4px;
        padding: 8px;
        font-size: 12px;
        color: #5e6c84;
        margin-bottom: 12px;
        border-left: 3px solid #0079bf;
      ">
        <strong>💡 Tip:</strong> Gebruik <strong>HH:MM</strong> formaat (bijv. 2:50 = 2.5 uur) of alleen cijfers (bijv. 1.5) voor de beste resultaten.
      </div>
      
      <button id="trello-hours-welcome-close-btn" style="
        background: #0079bf;
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 4px;
        font-size: 13px;
        font-weight: 500;
        cursor: pointer;
        width: 100%;
        transition: background-color 0.2s;
      " onmouseover="this.style.background='#005a8b'" onmouseout="this.style.background='#0079bf'">
        Begrepen!
      </button>
    `;
    
    document.body.appendChild(notification);
    // Add event listener for close button
    const closeBtn = notification.querySelector('#trello-hours-welcome-close-btn');
    if (closeBtn) {
      closeBtn.addEventListener('click', () => {
        notification.remove();
        localStorage.setItem('trello-hours-extension-version', currentVersion);
      });
    }
  }, 1000);
}

// Show welcome notification when page loads
showWelcomeNotification();
